import tkinter as tk
from tkinter import simpledialog, messagebox, ttk
import cv2
import os
import face_recognition
import numpy as np
from datetime import datetime
import openpyxl
import pickle
import requests
import pandas as pd

# Directories and files names
CAPTURE_DIR = "captured_images"
ENCODINGS_DIR = "trained_encodings"
DATA_DIR = "data"
EXCEL_FILE = os.path.join(DATA_DIR, "student_info.xlsx")
ATTENDANCE_FILE = os.path.join(DATA_DIR, "attendance.xlsx")

# Create necessary directories
os.makedirs(CAPTURE_DIR, exist_ok=True)
os.makedirs(ENCODINGS_DIR, exist_ok=True)
os.makedirs(DATA_DIR, exist_ok=True)

# Get approximate location (latitude,longitude) via IP geolocation
def get_location():
    try:
        response = requests.get("https://ipinfo.io/json")
        data = response.json()
        loc = data.get("loc", "0.0,0.0")
        return loc
    except:
        return "0.0,0.0"

# Capture multiple images of a student and store details
def capture_faces():
    name = simpledialog.askstring("Input", "Enter Name:")
    roll = simpledialog.askstring("Input", "Enter Roll Number:")
    class_name = simpledialog.askstring("Input", "Enter Class/Dept:")
    school = simpledialog.askstring("Input", "Enter School/Clg Name:")
    location = get_location()

    if not all([name, roll, class_name, school]):
        messagebox.showwarning("Missing Info", "Please fill all fields")
        return

    cam = cv2.VideoCapture(0)
    count = 0
    student_folder = os.path.join(CAPTURE_DIR, f"{name}_{roll}")
    os.makedirs(student_folder, exist_ok=True)

    messagebox.showinfo("Instructions", "Press SPACE to capture each image. Capture 10 images. Press ESC to stop early.")

    while True:
        ret, frame = cam.read()
        if not ret:
            break

        cv2.imshow("Capture Faces - SPACE to capture, ESC to quit", frame)
        key = cv2.waitKey(1)

        if key % 256 == 27:  # ESC key
            break
        elif key % 256 == 32:  # SPACE key
            img_path = os.path.join(student_folder, f"{name}_{count}.jpg")
            cv2.imwrite(img_path, frame)
            print(f"Captured {img_path}")
            count += 1
            if count >= 20:
                break

    cam.release()
    cv2.destroyAllWindows()

    timestamp = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
    save_student_info(name, roll, class_name, school, location, timestamp)

    messagebox.showinfo("Done", f"Captured {count} images for {name}")

# Save student info to Excel
def save_student_info(name, roll, class_name, school, location, timestamp):
    if not os.path.exists(EXCEL_FILE):
        wb = openpyxl.Workbook()
        ws = wb.active
        ws.append(["Name", "Roll", "Class", "School", "Location", "Timestamp"])
    else:
        wb = openpyxl.load_workbook(EXCEL_FILE)
        ws = wb.active

    # Avoid duplicate entries for same roll number
    rolls = [row[1].value for row in ws.iter_rows(min_row=2)]
    if roll not in rolls:
        ws.append([name, roll, class_name, school, location, timestamp])
        wb.save(EXCEL_FILE)
    else:
        messagebox.showinfo("Info", f"Roll number {roll} already registered.")

# Train face encodings from captured images
def train_faces():
    encodings = []
    labels = []

    for student_folder in os.listdir(CAPTURE_DIR):
        path = os.path.join(CAPTURE_DIR, student_folder)
        for file in os.listdir(path):
            img_path = os.path.join(path, file)
            image = face_recognition.load_image_file(img_path)
            face_locations = face_recognition.face_locations(image)
            if len(face_locations) == 0:
                continue
            encoding = face_recognition.face_encodings(image, known_face_locations=face_locations)[0]
            encodings.append(encoding)
            labels.append(student_folder)

    if not encodings:
        messagebox.showerror("Error", "No faces found to train.")
        return

    data = {"encodings": encodings, "labels": labels}
    with open(os.path.join(ENCODINGS_DIR, "encodings.pkl"), "wb") as f:
        pickle.dump(data, f)

    messagebox.showinfo("Success", "Training completed!")

# Recognize faces and mark attendance with present/absent in Excel
def recognize_faces():
    encoding_path = os.path.join(ENCODINGS_DIR, "encodings.pkl")
    if not os.path.exists(encoding_path):
        messagebox.showerror("Error", "Please train data first.")
        return

    with open(encoding_path, "rb") as f:
        data = pickle.load(f)

    cam = cv2.VideoCapture(0)
    location = get_location()

    if not os.path.exists(EXCEL_FILE):
        messagebox.showerror("Error", "No registered students found.")
        return

    student_wb = openpyxl.load_workbook(EXCEL_FILE)
    student_ws = student_wb.active
    student_data = []
    for row in student_ws.iter_rows(min_row=2, values_only=True):
        student_data.append({
            "name": row[0],
            "roll": row[1],
            "class": row[2],
            "school": row[3]
        })

    # Prepare attendance file: mark all absent initially
    if not os.path.exists(ATTENDANCE_FILE):
        wb = openpyxl.Workbook()
        ws = wb.active
        ws.append(["ID", "Name", "Roll", "Class", "School", "Status", "Timestamp", "Location"])
    else:
        wb = openpyxl.load_workbook(ATTENDANCE_FILE)
        ws = wb.active
        ws.delete_rows(2, ws.max_row)


    timestamp = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
    for idx, student in enumerate(student_data, start=1):
        ws.append([
            idx,
            student["name"],
            student["roll"],
            student["class"],
            student["school"],
            "Absent",
            timestamp,
            location
        ])

    wb.save(ATTENDANCE_FILE)

    recognized = set()

    messagebox.showinfo("Instructions", "Press ESC to stop recognition.")

    while True:
        ret, frame = cam.read()
        if not ret:
            break

        rgb = cv2.cvtColor(frame, cv2.COLOR_BGR2RGB)
        faces = face_recognition.face_locations(rgb)
        encodings_live = face_recognition.face_encodings(rgb, faces)

        for face_encoding, face_loc in zip(encodings_live, faces):
            matches = face_recognition.compare_faces(data["encodings"], face_encoding)
            face_distances = face_recognition.face_distance(data["encodings"], face_encoding)
            best_match_index = np.argmin(face_distances)

            if matches[best_match_index]:
                label = data["labels"][best_match_index]
                if label not in recognized:
                    try:
                        name, roll = label.rsplit("_", 1)
                    except:
                        name, roll = label, ""

                    for row in ws.iter_rows(min_row=2):
                        if row[1].value == name and row[2].value == roll:
                            row[5].value = "Present"
                            row[6].value = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
                            row[7].value = location
                            wb.save(ATTENDANCE_FILE)
                            break

                    recognized.add(label)

                top, right, bottom, left = face_loc
                cv2.rectangle(frame, (left, top), (right, bottom), (0, 255, 0), 2)
                cv2.putText(frame, label, (left, top - 10), cv2.FONT_HERSHEY_SIMPLEX, 0.8, (255, 255, 255), 2)

        cv2.imshow("Recognizing Faces - Press ESC to stop", frame)
        if cv2.waitKey(1) % 256 == 27:
            break

    wb.save(ATTENDANCE_FILE)
    cam.release()
    cv2.destroyAllWindows()
    messagebox.showinfo("Success", "Attendance updated with Present/Absent.")

# Show attendance info in new GUI table
from tkinter import filedialog

# Show attendance info in new GUI table
def info():
    if not os.path.exists(ATTENDANCE_FILE):
        messagebox.showerror("Error", "No attendance file found.")
        return

    df = pd.read_excel(ATTENDANCE_FILE)

    win1 = tk.Toplevel(root)
    win1.title("Attendance Info")
    win1.geometry("950x550")

    # --- Summary Section (Total, Present, Absent) ---
    total_students = len(df)
    present_count = (df["Status"] == "Present").sum()
    absent_count = (df["Status"] == "Absent").sum()

    summary_frame = tk.Frame(win1)
    summary_frame.pack(fill="x", pady=5)

    tk.Label(summary_frame, text=f"Total Students: {total_students}", font=("Arial", 12, "bold")).pack(side="left", padx=20)
    tk.Label(summary_frame, text=f"Present: {present_count}", font=("Arial", 12, "bold"), fg="green").pack(side="left", padx=20)
    tk.Label(summary_frame, text=f"Absent: {absent_count}", font=("Arial", 12, "bold"), fg="red").pack(side="left", padx=20)

    # --- Attendance Table ---
    tree = ttk.Treeview(win1, columns=list(df.columns), show="headings")

    for col in df.columns:
        tree.heading(col, text=col)
        tree.column(col, width=130, anchor="center")

    # Tag styles for Present/Absent
    tree.tag_configure("Present", background="lightgreen")
    tree.tag_configure("Absent", background="salmon")

    for _, row in df.iterrows():
        status = row["Status"]
        tag = "Present" if status == "Present" else "Absent"
        tree.insert("", "end", values=list(row), tags=(tag,))

    tree.pack(fill="both", expand=True)

    scrollbar = ttk.Scrollbar(win1, orient="vertical", command=tree.yview)
    tree.configure(yscroll=scrollbar.set)
    scrollbar.pack(side="right", fill="y")

    # --- Export Button ---
    def export_excel():
        file_path = filedialog.asksaveasfilename(
            defaultextension=".xlsx",
            filetypes=[("Excel files", "*.xlsx")],
            title="Save Attendance As"
        )
        if file_path:
            try:
                df.to_excel(file_path, index=False)
                messagebox.showinfo("Success", f"Attendance exported successfully!\n\nSaved at:\n{file_path}")
            except Exception as e:
                messagebox.showerror("Error", f"Failed to export: {e}")

    export_btn = tk.Button(win1, text="Export as Excel", command=export_excel, bg="lightblue", font=("Arial", 11, "bold"))
    export_btn.pack(pady=10)



# GUI setup
root = tk.Tk()
root.title("Face Attendance System")
root.geometry("400x380")

tk.Label(root, text="Face Attendance System", font=("Arial", 18)).pack(pady=10)

btn_capture = tk.Button(root, text="Capture Student Faces", command=capture_faces, width=30, height=2)
btn_capture.pack(pady=10)

btn_train = tk.Button(root, text="Train All Students", command=train_faces, width=30, height=2)
btn_train.pack(pady=10)

btn_recognize = tk.Button(root, text="Recognize & Mark Attendance", command=recognize_faces, width=30, height=2)
btn_recognize.pack(pady=10)

btn_info = tk.Button(root, text="View Attendance Info", command=info, width=30, height=2)
btn_info.pack(pady=10)

root.mainloop()

